package com.ejie.ab04b.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ejie.ab04b.constantes.Constantes;
import com.ejie.ab04b.constantes.ConstantesDokusi;
import com.ejie.ab04b.exception.AB04BParseException;
import com.ejie.ab04b.model.DocumentoAperturas;
import com.ejie.ab04b.model.DocumentoOS2;
import com.ejie.ab04b.model.DocumentoOS3;
import com.ejie.ab04b.parse.MessageParser;
import com.ejie.ab04b.parse.MessageUnparser;
import com.ejie.t65.jaxb.beans.fsd.T65BAttributeListType;
import com.ejie.t65.jaxb.beans.fsd.T65BAttributeType;
import com.ejie.t65.jaxb.beans.fsd.T65BConditionListType;
import com.ejie.t65.jaxb.beans.fsd.T65BConditionType;
import com.ejie.t65.jaxb.beans.fsd.T65BContentType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentIDListType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentIDType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentType;
import com.ejie.t65.jaxb.beans.fsd.T65BFormatListType;
import com.ejie.t65.jaxb.beans.fsd.T65BFormatType;
import com.ejie.t65.jaxb.beans.fsd.T65BSignatureInfoListType;
import com.ejie.t65.jaxb.beans.fsd.T65BSignatureInfoType;
import com.ejie.t65.jaxb.beans.fsd.T65BSignatureListType;
import com.ejie.t65.jaxb.beans.fsd.T65BSignatureType;
import com.ejie.t65.jaxb.beans.fsd.T65BStringListType;

/**
 * 
 * @author GFI-NORTE
 * 
 */
public final class DokusiUtils {

	private static DokusiUtils instance;

	private static final Logger LOGGER = LoggerFactory
			.getLogger(DokusiUtils.class);

	/**
	 * Constructor privado ya que ha de utilizarse el método getInstance().
	 */
	private DokusiUtils() {

	}

	/**
	 * Devuelve la instancia de tipo DokusiUtil.
	 * 
	 * Devuelve la instancia de tipo DokusiUtil
	 * 
	 * @return single instance of DokusiUtils
	 */
	public static final DokusiUtils getInstance() {
		if (DokusiUtils.instance == null) {
			DokusiUtils.instance = new DokusiUtils();
		}
		return DokusiUtils.instance;
	}

	/**
	 * A partir de un T65BContentType, devuelve su representación XML en un
	 * String.
	 * 
	 * t65bContentType objeto de tipo T65BContentType String e
	 * 
	 * @param t65bContentType
	 *            the t 65 b content type
	 * @return the t 65 B content type as string
	 */
	public String getT65BContentTypeAsString(T65BContentType t65bContentType) {
		MessageUnparser<T65BContentType> unparser = new MessageUnparser<T65BContentType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bContentType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bContentType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BConditionListType, devuelve su representación XML en
	 * un String.
	 * 
	 * t65BConditionListType objeto de tipo T65BConditionListType String e
	 * 
	 * @param t65BConditionListType
	 *            the t 65 B condition list type
	 * @return the t 65 B condition list type as string
	 */
	public String getT65BConditionListTypeAsString(
			T65BConditionListType t65BConditionListType) {
		MessageUnparser<T65BConditionListType> unparser = new MessageUnparser<T65BConditionListType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65BConditionListType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65BConditionListType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BFormatType, devuelve su representación XML en un
	 * String.
	 * 
	 * t65bFormatType objeto de tipo T65BFormatType String e
	 * 
	 * @param t65bFormatType
	 *            the t 65 b format type
	 * @return the t 65 B format type as string
	 */
	public String getT65BFormatTypeAsString(T65BFormatType t65bFormatType) {
		MessageUnparser<T65BFormatType> unparser = new MessageUnparser<T65BFormatType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bFormatType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bFormatType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BAttributeListType, devuelve su representación XML en
	 * un String.
	 * 
	 * t65bAttributeListType objeto de tipo T65BAttributeListType String e
	 * 
	 * @param t65bAttributeListType
	 *            the t 65 b attribute list type
	 * @return the t 65 B attribute list type as string
	 */
	public String getT65BAttributeListTypeAsString(
			T65BAttributeListType t65bAttributeListType) {
		MessageUnparser<T65BAttributeListType> unparser = new MessageUnparser<T65BAttributeListType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bAttributeListType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bAttributeListType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BStringListType, devuelve su representación XML en un
	 * String.
	 * 
	 * t65bStringListType objeto de tipo T65BStringListType String e
	 * 
	 * @param t65bStringListType
	 *            the t 65 b string list type
	 * @return the t 65 B string list type as string
	 */
	public String getT65BStringListTypeAsString(
			T65BStringListType t65bStringListType) {
		MessageUnparser<T65BStringListType> unparser = new MessageUnparser<T65BStringListType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bStringListType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bStringListType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BDocumentIDType, devuelve su representación XML en un
	 * String.
	 * 
	 * t65bDocumentIDType objeto de tipo T65BDocumentIDType String e
	 * 
	 * @param t65bDocumentIDType
	 *            the t 65 b document ID type
	 * @return the t 65 B document ID type as string
	 */
	public String getT65BDocumentIDTypeAsString(
			T65BDocumentIDType t65bDocumentIDType) {
		MessageUnparser<T65BDocumentIDType> unparser = new MessageUnparser<T65BDocumentIDType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bDocumentIDType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bDocumentIDType", e);
		}
		return resultado;
	}

	/**
	 * A partir de un T65BSignatureListType, devuelve su representación XML en
	 * un String.
	 * 
	 * t65bSignatureListType objeto de tipo T65BSignatureListType String e
	 * 
	 * @param t65bSignatureListType
	 *            the t 65 b signature list type
	 * @return the t 65 B signature list type as string
	 */
	public String getT65BSignatureListTypeAsString(
			T65BSignatureListType t65bSignatureListType) {
		MessageUnparser<T65BSignatureListType> unparser = new MessageUnparser<T65BSignatureListType>();
		String resultado = null;
		try {
			resultado = unparser.parse(t65bSignatureListType);
		} catch (Exception e) {
			DokusiUtils.LOGGER.error(
					"Error en el parseo del objeto t65bSignatureListType", e);
		}
		return resultado;
	}

	/**
	 * Devuelve un objeto T65BDocumentIDType a partir de su representación XML
	 * en String.
	 * 
	 * response String T65BDocumentIDType
	 * 
	 * @param response
	 *            the response
	 * @return the dokusi response as T 65 B document ID type
	 * @throws AB04BParseException
	 *             expcepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDType getDokusiResponseAsT65BDocumentIDType(
			String response) throws AB04BParseException {

		DokusiUtils.LOGGER.info("getDokusiResponseAsT65BDocumentIDType");

		T65BDocumentIDType obj = null;
		try {
			obj = (T65BDocumentIDType) new MessageParser<T65BDocumentIDType>(
					T65BDocumentIDType.class.getPackage().getName())
					.parse(response);
		} catch (Exception e) {
			DokusiUtils.LOGGER
					.error("Error en el parseo del objeto response como T65BDocumentIDType",
							e);

			throw new AB04BParseException(
					e,
					"LOG NEGOCIO: getDokusiResponseAsT65BDocumentIDType --> Error en el parseo del objeto response como T65BDocumentIDType");
		}

		return obj;
	}

	/**
	 * Devuelve un objeto T65BDocumentIDListType a partir de su representación
	 * XML en String.
	 * 
	 * response String T65BDocumentIDListType
	 * 
	 * @param response
	 *            the response
	 * @return the dokusi response as T 65 B document ID list type
	 * @throws AB04BParseException
	 *             expcepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDListType getDokusiResponseAsT65BDocumentIDListType(
			String response) throws AB04BParseException {

		DokusiUtils.LOGGER.info("getDokusiResponseAsT65BDocumentIDListType");

		T65BDocumentIDListType obj = null;
		try {
			obj = (T65BDocumentIDListType) new MessageParser<T65BDocumentIDListType>(
					T65BDocumentIDListType.class.getPackage().getName())
					.parse(response);
		} catch (Exception e) {
			DokusiUtils.LOGGER
					.error("Error en el parseo del objeto response como T65BDocumentIDListType",
							e);

			throw new AB04BParseException(
					e,
					" LOG NEGOCIO: getDokusiResponseAsT65BDocumentIDListType --> Error en el parseo del objeto response como T65BDocumentIDListType");
		}

		return obj;
	}

	/**
	 * Devuelve un objeto T65BDocumentIDType a partir de su representación XML
	 * en String.
	 * 
	 * response String T65BDocumentType
	 * 
	 * @param response
	 *            the response
	 * @return the dokusi response as T 65 B document type
	 * @throws AB04BParseException
	 *             expcepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentType getDokusiResponseAsT65BDocumentType(String response)
			throws AB04BParseException {

		DokusiUtils.LOGGER.info("getDokusiResponseAsT65BDocumentType");

		T65BDocumentType obj = null;
		try {
			obj = (T65BDocumentType) new MessageParser<T65BDocumentType>(
					T65BDocumentIDType.class.getPackage().getName())
					.parse(response);
		} catch (Exception e) {
			DokusiUtils.LOGGER
					.error("Error en el parseo del objeto response como T65BDocumentType",
							e);

			throw new AB04BParseException(
					e,
					" LOG NEGOCIO: getDokusiResponseAsT65BDocumentType --> Error en el parseo del objeto response como T65BDocumentType");
		}

		return obj;
	}

	/**
	 * Crea un objeto T65BContentType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BContentType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B content type
	 */
	public T65BContentType construirT65BContentType(DocumentoAperturas doc) {
		T65BContentType obj = new T65BContentType();
		obj.setFormat(this.construirT65BFormatType(doc));
		obj.setPifId(doc.getRutaPif065());
		return obj;
	}

	/**
	 * Crea un objeto T65BContentType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BContentType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B content type
	 */
	public T65BContentType construirT65BContentType(DocumentoOS2 doc) {
		T65BContentType obj = new T65BContentType();
		obj.setFormat(this.construirT65BFormatType(doc));
		obj.setPifId(doc.getRutaPif094());
		return obj;
	}

	/**
	 * Crea un objeto T65BContentType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BContentType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B content type
	 */
	public T65BContentType construirT65BContentType(DocumentoOS3 doc) {
		T65BContentType obj = new T65BContentType();
		obj.setFormat(this.construirT65BFormatType(doc));
		obj.setPifId(doc.getRutaPif097());
		return obj;
	}

	/**
	 * Crea un objeto T65BFormatType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BFormatType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B format type
	 */
	public T65BFormatType construirT65BFormatType(DocumentoAperturas doc) {
		T65BFormatType obj = new T65BFormatType();
		obj.setExtension(Utilities.getInstance().obtenerExtensionFichero(
				doc.getRutaPif065()));
		obj.setName(doc.getDokusiFormat());
		return obj;
	}

	/**
	 * Crea un objeto T65BFormatType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BFormatType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B format type
	 */
	public T65BFormatType construirT65BFormatType(DocumentoOS2 doc) {
		T65BFormatType obj = new T65BFormatType();
		obj.setExtension(Utilities.getInstance().obtenerExtensionFichero(
				doc.getRutaPif094()));
		obj.setName(doc.getDokusiFormat());
		return obj;
	}

	/**
	 * Crea un objeto T65BFormatType a partir de sus atributos.
	 * 
	 * doc DocumentoAperturas T65BFormatType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B format type
	 */
	public T65BFormatType construirT65BFormatType(DocumentoOS3 doc) {
		T65BFormatType obj = new T65BFormatType();
		obj.setExtension(Utilities.getInstance().obtenerExtensionFichero(
				doc.getRutaPif097()));
		obj.setName(doc.getDokusiFormat());
		return obj;
	}

	/**
	 * Crea un objeto T65BAttributeType a partir de sus atributos.
	 * 
	 * key String value String T65BAttributeType
	 * 
	 * @param key
	 *            the key
	 * @param value
	 *            the value
	 * @return the t 65 B attribute type
	 */
	public T65BAttributeType construirT65BAttributeType(String key, String value) {
		T65BAttributeType obj = new T65BAttributeType();
		obj.setKey(key);
		obj.setValue(value);
		return obj;
	}

	/**
	 * Crea un objeto T65BAttributeType a partir de sus atributos.
	 * 
	 * attribute T65BAttributeType operator String T65BAttributeType
	 * 
	 * @param attribute
	 *            the attribute
	 * @param operator
	 *            the operator
	 * @return the t 65 B condition type
	 */
	public T65BConditionType construirT65BConditionType(
			T65BAttributeType attribute, String operator) {
		T65BConditionType obj = new T65BConditionType();
		obj.setAttribute(attribute);
		obj.setOperator(operator);
		return obj;
	}

	/**
	 * Genera la lista de metadatos básicos de un documento.
	 * 
	 * doc DocumentoAperturas T65BAttributeListType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B attribute list type
	 */
	public T65BAttributeListType generarMetadatosBasicos(DocumentoAperturas doc) {
		Map<String, String> metadatos = new HashMap<String, String>();

		metadatos.put(ConstantesDokusi.META_NOMBRE, doc.getNombreDoc065());
		metadatos.put(ConstantesDokusi.META_TITULO, doc.getNombreDoc065());

		if (doc.getTramiteAperturas() == null
				|| doc.getTramiteAperturas().getApertura() == null) {
			metadatos.put(ConstantesDokusi.META_SERIE,
					this.obtenerSerieDocumentalAperturas(doc.getTerritorio()));
		} else {
			metadatos.put(ConstantesDokusi.META_SERIE, this
					.obtenerSerieDocumentalAperturas(doc.getTramiteAperturas()
							.getApertura().getTerape056()));
		}

		metadatos.put(ConstantesDokusi.META_PROCEDIMIENTO, Utilities
				.getInstance().obtenerDaProcedureIdApertura());
		metadatos.put(ConstantesDokusi.META_ORIGEN,
				this.obtenerMetadatoOrigenDokusi(doc.getOrigen065()));
		metadatos.put(ConstantesDokusi.META_ACL,
				PropertiesUtil.getProperty(PropertiesUtil.ACL_DOKUSI));

		return this.construirT65BAttributeListType(metadatos);
	}

	/**
	 * Genera la lista de metadatos básicos de un documento.
	 * 
	 * doc DocumentoAperturas T65BAttributeListType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B attribute list type
	 */
	public T65BAttributeListType generarMetadatosBasicos(DocumentoOS2 doc) {
		Map<String, String> metadatos = new HashMap<String, String>();

		metadatos.put(ConstantesDokusi.META_NOMBRE, doc.getNombreDoc094());
		metadatos.put(ConstantesDokusi.META_TITULO, doc.getNombreDoc094());
		metadatos.put(
				ConstantesDokusi.META_SERIE,
				this.obtenerSerieDocumentalOS(doc.getTramiteOS2().getos2()
						.getTeros2060()));
		metadatos.put(ConstantesDokusi.META_PROCEDIMIENTO, Utilities
				.getInstance().obtenerDaProcedureIdOs2());
		metadatos.put(ConstantesDokusi.META_ORIGEN,
				this.obtenerMetadatoOrigenDokusi(doc.getOrigen094()));
		metadatos.put(ConstantesDokusi.META_ACL,
				PropertiesUtil.getProperty(PropertiesUtil.ACL_DOKUSI));

		return this.construirT65BAttributeListType(metadatos);
	}

	/**
	 * Genera la lista de metadatos básicos de un documento.
	 * 
	 * doc DocumentoAperturas T65BAttributeListType
	 * 
	 * @param doc
	 *            the doc
	 * @return the t 65 B attribute list type
	 */
	public T65BAttributeListType generarMetadatosBasicos(DocumentoOS3 doc) {
		Map<String, String> metadatos = new HashMap<String, String>();

		metadatos.put(ConstantesDokusi.META_NOMBRE, doc.getNombreDoc097());
		metadatos.put(ConstantesDokusi.META_TITULO, doc.getNombreDoc097());
		metadatos.put(
				ConstantesDokusi.META_SERIE,
				this.obtenerSerieDocumentalOS(doc.getTramiteOS3().getOs3()
						.getTeros3061()));
		metadatos.put(ConstantesDokusi.META_PROCEDIMIENTO, Utilities
				.getInstance().obtenerDaProcedureIdOs3());
		metadatos.put(ConstantesDokusi.META_ORIGEN,
				this.obtenerMetadatoOrigenDokusi(doc.getOrigen097()));
		metadatos.put(ConstantesDokusi.META_ACL,
				PropertiesUtil.getProperty(PropertiesUtil.ACL_DOKUSI));

		return this.construirT65BAttributeListType(metadatos);
	}

	/**
	 * Obtener serie documental aperturas.
	 * 
	 * territorio String String
	 * 
	 * @param territorio
	 *            the territorio
	 * @return the string
	 */
	public String obtenerSerieDocumentalAperturas(String territorio) {

		StringBuilder key = new StringBuilder(
				PropertiesUtil.SERIE_DOCUMENTAL_DOKUSI_APERTURAS).append(
				Constantes.SEPARADOR_PUNTO).append(territorio);
		return PropertiesUtil.getProperty(key.toString());
	}

	/**
	 * Obtener serie documental OS 2.
	 * 
	 * territorio String String
	 * 
	 * @param territorio
	 *            the territorio
	 * @return the string
	 */
	public String obtenerSerieDocumentalOS2(String territorio) {

		StringBuilder key = new StringBuilder(
				PropertiesUtil.SERIE_DOCUMENTAL_DOKUSI_OS).append(
				Constantes.SEPARADOR_PUNTO).append(territorio);
		return PropertiesUtil.getProperty(key.toString());
	}

	/**
	 * Obtener serie documental OS.
	 * 
	 * territorio String String
	 * 
	 * @param territorio
	 *            the territorio
	 * @return the string
	 */
	public String obtenerSerieDocumentalOS(String territorio) {

		StringBuilder key = new StringBuilder(
				PropertiesUtil.SERIE_DOCUMENTAL_DOKUSI_OS).append(
				Constantes.SEPARADOR_PUNTO).append(territorio);
		return PropertiesUtil.getProperty(key.toString());
	}

	/**
	 * Devuelve el valor del metadato de Dokusi origen en función del valor
	 * almacenado en negocio.
	 * 
	 * origen Integer String
	 * 
	 * @param origen
	 *            the origen
	 * @return the string
	 */
	public String obtenerMetadatoOrigenDokusi(Integer origen) {
		if (origen != null) {
			if (origen.equals(Constantes.DOCUMENTO_ORIGEN_CIUDADANO)) {
				return ConstantesDokusi.META_ORIGEN_CIUDADANIA;
			} else if (origen
					.equals(Constantes.DOCUMENTO_ORIGEN_ADMINISTRACION)) {
				return ConstantesDokusi.META_ORIGEN_ADMINISTRACION;
			}
		}
		return ConstantesDokusi.META_ORIGEN_CIUDADANIA;
	}

	/**
	 * Genera la lista de condiciones de consulta de documentos de una
	 * solicitud.
	 * 
	 * numRegistro String procedureId String T65BAttributeListType
	 * 
	 * @param numRegistro
	 *            the num registro
	 * @param procedureId
	 *            the procedure id
	 * @return the t 65 B condition list type
	 */
	public T65BConditionListType generarCondicionesConsulta(String numRegistro,
			String procedureId) {
		T65BConditionListType conditionList = new T65BConditionListType();

		T65BAttributeType attributeNumRegistro = this
				.construirT65BAttributeType(ConstantesDokusi.META_NUM_REGISTRO,
						numRegistro);
		T65BConditionType conditionNumRegistro = this
				.construirT65BConditionType(attributeNumRegistro,
						Constantes.OPERADOR_IGUAL);

		T65BAttributeType attributeACL = this.construirT65BAttributeType(
				ConstantesDokusi.META_ACL,
				PropertiesUtil.getProperty(PropertiesUtil.ACL_DOKUSI));
		T65BConditionType conditionACL = this.construirT65BConditionType(
				attributeACL, Constantes.OPERADOR_IGUAL);

		conditionList.getCondition().add(conditionNumRegistro);
		conditionList.getCondition().add(conditionACL);

		return conditionList;
	}

	/**
	 * Crea un objeto T65BDocumentIDType a partir de sus atributos.
	 * 
	 * id String version String T65BDocumentIDType
	 * 
	 * @param id
	 *            the id
	 * @param version
	 *            the version
	 * @return the t 65 B document ID type
	 */
	public T65BDocumentIDType construirT65BDocumentIDType(String id,
			String version) {
		T65BDocumentIDType obj = new T65BDocumentIDType();
		obj.setId(id);
		obj.setVersion(version);
		return obj;
	}

	/**
	 * Crea un objeto T65BAttributeListType a partir de un mapa de metadatos.
	 * 
	 * metadatos Map<String, String> T65BAttributeListType
	 * 
	 * @param metadatos
	 *            the metadatos
	 * @return the t 65 B attribute list type
	 */
	public T65BAttributeListType construirT65BAttributeListType(
			Map<String, String> metadatos) {
		T65BAttributeListType t65bAttributeListType = new T65BAttributeListType();
		if (metadatos != null) {
			Set<String> keys = metadatos.keySet();
			Iterator<String> iterator = keys.iterator();
			while (iterator.hasNext()) {
				String key = iterator.next();
				String value = metadatos.get(key);
				T65BAttributeType t65bAttributeType = this
						.construirT65BAttributeType(key, value);
				t65bAttributeListType.getAttribute().add(t65bAttributeType);
			}
		}
		return t65bAttributeListType;
	}

	/**
	 * Crea un objeto T65BSignatureType a partir de sus atributos.
	 * 
	 * signedDocumentId T65BDocumentIDType signature String signatureInfo
	 * T65BSignatureType date String type String isConservable String
	 * ejgv_cons_firma_longevo documentIsRequiered String
	 * ejgv_req_doc_firma_longevo signatureType String signatureFormat String
	 * signatureProfile String timestampSerialNumber String format
	 * T65BFormatType T65BSignatureType
	 * 
	 * @param signedDocumentId
	 *            the signed document id
	 * @param signature
	 *            the signature
	 * @param signatureInfo
	 *            the signature info
	 * @param date
	 *            the date
	 * @param type
	 *            the type
	 * @param isConservable
	 *            the is conservable
	 * @param documentIsRequiered
	 *            the document is requiered
	 * @param signatureType
	 *            the signature type
	 * @param signatureFormat
	 *            the signature format
	 * @param signatureProfile
	 *            the signature profile
	 * @param timestampSerialNumber
	 *            the timestamp serial number
	 * @param format
	 *            the format
	 * @return the t 65 B signature type
	 */
	public T65BSignatureType construirT65BSignatureType(
			T65BDocumentIDType signedDocumentId, String signature,
			T65BSignatureInfoType signatureInfo, String date, String type,
			String isConservable, String documentIsRequiered,
			String signatureType, String signatureFormat,
			String signatureProfile, String timestampSerialNumber,
			T65BFormatType format) {
		T65BSignatureType obj = new T65BSignatureType();
		obj.setSignedDocumentId(signedDocumentId);
		obj.setSignature(signature);

		T65BSignatureInfoListType t65bSignatureInfoListType = new T65BSignatureInfoListType();
		t65bSignatureInfoListType.getSignatureInfo().add(signatureInfo);
		obj.setSignatureInfoList(t65bSignatureInfoListType);

		obj.setDate(date);
		obj.setType(type);
		obj.setIsConservable(isConservable);
		obj.setDocumentIsRequiered(documentIsRequiered);
		obj.setSignatureType(signatureType);
		obj.setSignatureFormat(signatureFormat);
		obj.setSignatureProfile(signatureProfile);
		obj.setTimestampSerialNumber(timestampSerialNumber);
		obj.setFormat(format);
		return obj;
	}

	/**
	 * Crea un objeto T65BSignatureInfoType a partir de sus atributos.
	 * 
	 * department String entity String responsability String signerID String
	 * signerName String T65BSignatureInfoType
	 * 
	 * @param department
	 *            the department
	 * @param entity
	 *            the entity
	 * @param responsability
	 *            the responsability
	 * @param signerID
	 *            the signer ID
	 * @param signerName
	 *            the signer name
	 * @return the t 65 B signature info type
	 */
	public T65BSignatureInfoType construirT65BSignatureInfoType(
			String department, String entity, String responsability,
			String signerID, String signerName) {
		T65BSignatureInfoType obj = new T65BSignatureInfoType();
		obj.setDepartment(department);
		obj.setEntity(entity);
		obj.setResponsability(responsability);
		obj.setSignerID(signerID);
		obj.setSignerName(signerName);
		return obj;
	}

	/**
	 * Se obtiene solo el contenido del objeto, no su tipo, cabecera, namespace,
	 * etc.
	 * 
	 * firma Firma sin tratar String firma tratada
	 * 
	 * @param firma
	 *            the firma
	 * @return the string
	 */
	public String tratarFirmaParaRegistrarEnDokusi(String firma) {

		if (firma == null || firma.trim().length() == 0) {
			return firma;
		}
		int i = firma.indexOf("<ejgvDocument>");
		String firmaTratada = firma;
		if (i > 0) {
			firmaTratada = firma.substring(i);
			i = firmaTratada.indexOf("</ejgvDocument>");
			if (i > 0) {
				i += "</ejgvDocument>".length();
				firmaTratada = firmaTratada.substring(0, i);
			}
		}

		return firmaTratada;
	}

	/**
	 * Crea un objeto T65BFormatType para transformacion a PDFA.
	 * 
	 * T65BFormatType
	 * 
	 * @return the t 65 B format type
	 */
	public T65BFormatType construirT65BFormatTypePDFA() {
		T65BFormatType obj = new T65BFormatType();
		obj.setExtension(Constantes.EXT_PDF);
		obj.setFullText(null);
		obj.setName(Constantes.EXT_PDFA);
		return obj;
	}

	/**
	 * Obtener metadato lista.
	 * 
	 * t65BAttributeListType T65BAttributeListType metadato String the string
	 * 
	 * @param t65BAttributeListType
	 *            the t 65 B attribute list type
	 * @param metadato
	 *            the metadato
	 * @return the string
	 */
	public String obtenerMetadatoLista(
			T65BAttributeListType t65BAttributeListType, String metadato) {
		if (t65BAttributeListType != null
				&& t65BAttributeListType.getAttribute() != null) {
			for (T65BAttributeType attr : t65BAttributeListType.getAttribute()) {
				if (metadato.equals(attr.getKey())) {
					return attr.getValue();
				}
			}
		}

		return null;
	}

	/**
	 * Devuelve un objeto T65BDocumentIDType a partir de su representación XML
	 * en String.
	 * 
	 * response String T65BDocumentType
	 * 
	 * @param response
	 *            the response
	 * @return the dokusi response as T 65 B format list type
	 * @throws AB04BParseException
	 *             expcepción al parsear la respuesta de Dokusi
	 */
	public T65BFormatListType getDokusiResponseAsT65BFormatListType(
			String response) throws AB04BParseException {

		DokusiUtils.LOGGER.info("getDokusiResponseAsT65BFormatListType");

		T65BFormatListType obj = null;
		try {
			obj = (T65BFormatListType) new MessageParser<T65BFormatListType>(
					T65BFormatListType.class.getPackage().getName())
					.parse(response);
		} catch (Exception e) {
			DokusiUtils.LOGGER
					.error("Error en el parseo del objeto response como T65BFormatListType",
							e);

			throw new AB04BParseException(
					e,
					" LOG NEGOCIO: getDokusiResponseAsT65BFormatListType --> Error en el parseo del objeto response como T65BFormatListType");
		}

		return obj;
	}

}
